/*
 * FMA related alerts
 */
#include "libfma.h"
#include "lf_alert.h"
#include "lf_fabric.h"

#include "fms.h"
#include "fms_alert.h"
#include "fms_fabric.h"
#include "fms_error.h"
#include "fms_settings.h"

/*
 * Initialize alert structs in host
 */
void
fms_fma_init_host_alerts(
  struct lf_host *hp)
{
  struct fms_alert *ap;

  /* allocate and initialize hosts alert anchor */
  LF_CALLOC(ap, struct fms_alert, 1);
  fms_init_alert_anchor(ap);

  FMS_HOST(hp)->alerts_anchor = ap;

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Post an alert indicating no contact from this host.
 * Given the nature of how this alert is generated, it is not
 * possible for a pre-existing duplicate event to exist.
 */
void
fms_fma_alert_no_initial_fma(
  struct lf_host *hp)
{
  struct fms_alert *ap;

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_HOST_NO_INITIAL_FMA);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in hostname */
  LF_STRCPY(ap->alert.a.host_no_initial_fma.hostname, hp->hostname);

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_HOST(hp)->alerts_anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Post an alert indicating loss of contact with this FMA
 */
void
fms_fma_alert_lost_fma(
  struct lf_host *hp)
{
  struct fms_alert *ap;

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_HOST_LOST_FMA);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in hostname */
  LF_STRCPY(ap->alert.a.host_lost_fma.hostname, hp->hostname);

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_HOST(hp)->alerts_anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Got an attachment from an FMA.  We need to relic any outstanding
 * alerts.
 */
void
fms_fma_alert_present(
  struct lf_host *hp)
{
  lf_alert_type_t types[2];

  /* relic all alerts of either type */
  types[0] = LF_ALERT_HOST_NO_INITIAL_FMA;
  types[1] = LF_ALERT_HOST_LOST_FMA;

  fms_alert_relic_matches_in_subj_list(FMS_HOST(hp)->alerts_anchor, types, 2);
}

/*
 * SRAM parity error seen on a NIC, report it
 */
void
fms_fma_alert_sram_parity_error(
  struct lf_host *hp,
  int nic_id)
{
  struct fms_alert *ap;

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_HOST_SRAM_PARITY_ERROR);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in hostname, et. al. */
  LF_STRCPY(ap->alert.a.host_sram_parity_error.hostname, hp->hostname);
  ap->alert.a.host_sram_parity_error.nic_id = nic_id;
  if (nic_id <= hp->num_nics) {
    LF_STRCPY(ap->alert.a.host_sram_parity_error.serial_no,
	hp->nics[nic_id]->serial_no);
  } else {
    LF_STRCPY(ap->alert.a.host_sram_parity_error.serial_no,
	"<bad nic_id>");
  }

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_HOST(hp)->alerts_anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * NIC stopped responding, report it
 */
void
fms_fma_alert_nic_died(
  struct lf_host *hp,
  int nic_id)
{
  struct fms_alert *ap;

  /* get an alert struct */
  ap = fms_allocate_alert(LF_ALERT_HOST_FIRMWARE_DIED);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in hostname, et. al. */
  LF_STRCPY(ap->alert.a.host_firmware_died.hostname, hp->hostname);
  ap->alert.a.host_firmware_died.nic_id = nic_id;
  if (nic_id <= hp->num_nics) {
    LF_STRCPY(ap->alert.a.host_firmware_died.serial_no,
	hp->nics[nic_id]->serial_no);
  } else {
    LF_STRCPY(ap->alert.a.host_firmware_died.serial_no,
	"<bad nic_id>");
  }

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_HOST(hp)->alerts_anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * NIC has unrecognized type
 */
void
fms_fma_alert_unrecognized_nic_type(
  struct lf_host *hp,
  int nic_id,
  lf_string_t product_id)
{
  struct fms_alert *ap;
  struct host_unrecognized_nic_type_struct *adata;
  int this_alert;
  struct fms_alert *anchor;

  /*
   * First, do the ugly work of looking for an alert of this type in the
   * link's list of alerts.  If we find an active alert already,
   * just return.
   */
  this_alert = LF_ALERT_HOST_UNRECOGNIZED_NIC_TYPE;
  anchor = FMS_HOST(hp)->alerts_anchor;
  ap = anchor->subj_next;
  while (ap != anchor) {
    if (ap->alert.alert_type == this_alert) {
      adata = &ap->alert.a.host_unrecognized_nic_type;
      if (adata->nic_id == nic_id) {
	return;
      }
    }
    ap = ap->subj_next;
  }

  /* get an alert struct */
  ap = fms_allocate_alert(this_alert);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in hostname, et. al. */
  adata = &ap->alert.a.host_unrecognized_nic_type;
  LF_STRCPY(adata->hostname, hp->hostname);
  LF_STRCPY(adata->product_id, product_id);
  adata->nic_id = nic_id;

  /* register the alert */
  fms_register_alert(ap);

  /* link it into host list */
  fms_subj_link_alert(FMS_HOST(hp)->alerts_anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * A link between an enclosure and a host has a high badcrc count from
 * the host side
 */
void
fms_fma_alert_nic_badcrc(
  struct lf_nic *nicp,
  int port,
  int badcrcs)
{
  struct fms_settings *fsp;
  struct fms_alert *ap;
  struct lf_xcvr *xcp1;
  int xcport1;
  struct lf_xcvr *xcp2;
  int xcport2;
  struct lf_host *hp;
  int this_alert;
  struct fms_alert *anchor;
  struct host_switch_link_badcrc_count_struct *adata;

  fsp = F.settings;

  /* get xcvr for this end */
  xcp1 = LF_XCVR(nicp->phys_ports[port]);
  xcport1 = nicp->phys_rports[port];

  /* get the other end */
  xcp2 = LF_XCVR(xcp1->ports[xcport1 - xcp1->num_conns]);
  xcport2 = xcp1->rports[xcport1 - xcp1->num_conns];

  /* make sure the other end is what we expect */
  if (xcp2->ln_type != LF_NODE_LC_XCVR) {
    LF_ERROR(("Other end of phys conn must be linecard xcvr"));
  }

  /*
   * First, do the ugly work of looking for an alert of this type in the
   * link's list of alerts.  If we find an active alert already,
   * just return.
   */
  hp = nicp->host;
  this_alert = LF_ALERT_HOST_SWITCH_LINK_BADCRC_COUNT;
  anchor = FMS_HOST(hp)->alerts_anchor;
  ap = anchor->subj_next;
  while (ap != anchor) {
    if (ap->alert.alert_type == this_alert) {
      adata = &ap->alert.a.host_switch_link_badcrc_count;
      if (adata->nic == nicp->host_nic_id && adata->nic_interface == port) {
	return;
      }
    }
    ap = ap->subj_next;
  }

  /* get an alert struct */
  ap = fms_allocate_alert(this_alert);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  adata = &ap->alert.a.host_switch_link_badcrc_count;

  LF_STRCPY(adata->hostname, hp->hostname);
  adata->nic = nicp->host_nic_id;
  adata->nic_interface = port;

  LF_STRCPY(adata->enclosure, xcp2->p.linecard->enclosure->name);
  adata->slot = lf_slot_display_no(xcp2->p.linecard);
  adata->port = xcp2->p.linecard->xcvr_labels[xcp2->port];
  adata->subport = xcport2;

  adata->badcrc_count = badcrcs;
  adata->seconds = fsp->nic_query_interval;

  /* register the alert */
  fms_register_alert(ap);

  /* link it into xbar list */
  fms_subj_link_alert(anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * A port on a NIC is down and the other end is not known
 */
void
fms_fma_alert_nic_port_down(
  struct lf_nic *nicp,
  int port)
{
  struct fms_alert *ap;
  struct lf_host *hp;
  int this_alert;
  struct fms_alert *anchor;
  struct host_port_down_struct *adata;

  /*
   * First, do the ugly work of looking for an alert of this type in the
   * link's list of alerts.  If we find an active alert already,
   * just return.
   */
  hp = nicp->host;
  this_alert = LF_ALERT_HOST_PORT_DOWN;
  anchor = FMS_HOST(hp)->alerts_anchor;
  ap = anchor->subj_next;
  while (ap != anchor) {
    if (ap->alert.alert_type == this_alert) {
      adata = &ap->alert.a.host_port_down;
      if (adata->nic == nicp->host_nic_id && adata->nic_port == port) {
	return;
      }
    }
    ap = ap->subj_next;
  }

  /* get an alert struct */
  ap = fms_allocate_alert(this_alert);
  if (ap == NULL) LF_ERROR(("Error allocating alert"));

  /* Fill in the alert */
  adata = &ap->alert.a.host_port_down;

  LF_STRCPY(adata->hostname, hp->hostname);
  adata->nic = nicp->host_nic_id;
  adata->nic_port = port;

  /* register the alert */
  fms_register_alert(ap);

  /* link it into xbar list */
  fms_subj_link_alert(anchor, ap);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * NIC port is no longer down
 */
void
fms_fma_alert_nic_port_up(
  struct lf_nic *nicp,
  int port)
{
  struct fms_alert *ap;
  struct fms_alert *nap;
  struct lf_host *hp;
  int this_alert;
  struct fms_alert *anchor;
  struct host_port_down_struct *adata;

  /*
   * Find a matching alert and make it a relic
   */
  hp = nicp->host;
  this_alert = LF_ALERT_HOST_PORT_DOWN;
  anchor = FMS_HOST(hp)->alerts_anchor;
  ap = anchor->subj_next;
  while (ap != anchor) {
    nap = ap->subj_next;	/* be safe */

    if (ap->alert.alert_type == this_alert) {
      adata = &ap->alert.a.host_port_down;
      if (adata->nic == nicp->host_nic_id && adata->nic_port == port) {
	fms_relic_alert(ap);
	return;
      }
    }
    ap = nap;
  }
}
